// This shader is using the alpha channel to draw the textures as layers on top of each other.
// Lower alpha in texture channel 2 using UV2 shows more of texture channel 1 using world space position.
// Lower alpha in texture channel 1 using world space position shows more of texture channel 0 also using world space position but in another scale.

// Cutting plane and fog has been disabled to save performance

// Game specific constants
#define LargeFloorTextureScale 10.0f
#define SmallFloorTextureScale 2.0f

// You may modify this to move more data from the vertex shader to the pixel shader.
// It needs at least the position to work.
struct VS_to_PS {
	float4 Pos : SV_POSITION;
	float4 Tex : TEXCOORD0;
	float4 Color : COLOR;
	float4 Pos_CameraSpace : TEXCOORD1;
	float3 Normal_WorldSpace : NORMAL;
	float4 Pos_WorldSpace : TEXCOORD2;
};

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
VS_to_PS VS( VS_structure input ) {
	VS_to_PS output = (VS_to_PS)0;
	
	// Convert vertice positions from object space to world space
	output.Pos_WorldSpace = mul( input.Pos, ObjectToWorld );
	// Convert vertice positions from world space to camera space
	output.Pos_CameraSpace = mul( output.Pos_WorldSpace, WorldToCamera );
	// Convert vertice positions from camera space to 2D projection
	output.Pos = mul( output.Pos_CameraSpace, CameraToImage );
	
	// Transform normals to world space
	output.Normal_WorldSpace = mul(input.Normal,(float3x3)ObjectToWorld);
	
	// Give the texture coordinates to the pixel shader
	output.Tex = float4(output.Pos_WorldSpace.xz,input.Tex.zw);
	
	// Give the vertex color multiplied with the instance color to the pixel shader
	output.Color = lerp(input.Color * InstanceColor, float4(0.0f,0.0f,1.0f,1.0f), input.Selected);
	
	return output;
}

//tex_0 is the bottom texture
//tex_1 is the middle texture
//tex_2 is the top texture

//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS( VS_to_PS input) : SV_Target {
	// If the camera has a world space cutting plane, clip on the negative side.
	//UseCuttingPlane(input.Pos_WorldSpace)
	
	float4 BottomColor;
	float4 MiddleColor; // Using world space x,z projection
	float4 TopColor;
	BottomColor = tex_0.Sample( samAnisotropicMipmap, input.Tex.xy / SmallFloorTextureScale );
	MiddleColor = tex_1.Sample( samAnisotropicMipmap, input.Tex.xy / LargeFloorTextureScale );
	TopColor = tex_2.Sample( samAnisotropicMipmap, input.Tex.zw );
	float4 finalColor = lerp(lerp(BottomColor,MiddleColor,MiddleColor.w),TopColor,TopColor.w) * input.Color;
	
	float3 DiffuseLight;
	float3 SpecularLight;
	Engine_GetDiffuseAndSpecularLight(input.Pos_WorldSpace,normalize(input.Normal_WorldSpace),30,DiffuseLight,SpecularLight);
	finalColor = (finalColor * float4(DiffuseLight + SpecularLight,1.0f));
	
	// Show the final color with fog on R,G,B channels
	//return float4((lerp(finalColor,FogColor,saturate(length(input.Pos_CameraSpace) / FogDistance) * MaxFogIntensity)).xyz,finalColor.w);
	return finalColor;
}
