
// Like M_Fog but only sample light at depth 0.
// The depth is the length of the object's X axis divided by 4.
// The target draw surface must have an extra depth buffer for this to work.

// You may modify this to move more data from the vertex shader to the pixel shader.
// It needs at least the position to work.
struct VS_to_PS {
	float4 Pos : SV_POSITION;
	float4 Tex : TEXCOORD0;
	float4 Color : COLOR;
	float4 Pos_WorldSpace : TEXCOORD1;
	float4 Pos_CameraSpace : TEXCOORD2;
};

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
VS_to_PS VS( VS_structure input ) {
	VS_to_PS output = (VS_to_PS)0;
	
	// Convert vertice positions from object space to world space
	output.Pos_WorldSpace = mul( input.Pos, ObjectToWorld );
	// Convert vertice positions from world space to camera space
	output.Pos_CameraSpace = mul( output.Pos_WorldSpace, WorldToCamera );
	
	// Convert vertice positions from camera space to 2D projection
	output.Pos = mul( output.Pos_CameraSpace, CameraToImage );
	
	// Give the texture coordinates to the pixel shader
	output.Tex = input.Tex;
	
	// Give the vertex color multiplied with the instance color to the pixel shader
	output.Color = lerp(input.Color * InstanceColor, float4(0.0f,0.0f,1.0f,1.0f), input.Selected);
	
	return output;
}

//tex_0 is the diffuse texture

//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS( VS_to_PS input) : SV_Target {
	// If the camera has a world space cutting plane, clip on the negative side.
	UseCuttingPlane(input.Pos_WorldSpace)
	
	// Sample the diffuse map and opacity
	float4 finalColor = tex_0.Sample( samAnisotropicMipmap, input.Tex.xy ) * input.Color;
	
	// sample global light
	float3 LightColor;
	Engine_GetUndirectedLight(input.Pos_WorldSpace, LightColor);
	
	// Sample a copy of the depth map and convert from ZW to linear depth using the current camera's projection matrix
	float DepthBufferLinear = CameraToImage._43 / (tex_depthBufferCopy.Sample( samConstant, input.Pos.xy / Dimensions ) - CameraToImage._33);
	
	// Use the length of the object's X axis divided by 4 as the thickness
	float Thickness = length(float3(ObjectToWorld._11,ObjectToWorld._12,ObjectToWorld._13)) / 4;
	
	// Fade alpha when the pixel is close to the background
	float CloseToBackgroundFading = saturate((DepthBufferLinear - input.Pos_CameraSpace.z) / Thickness);
	
	// Fade alpha when the pixel is close to the camera
	float CloseToCameraFading = saturate((input.Pos_CameraSpace.z - NearClipPlane) / Thickness);
	
	// Combine the result of the light samples and the fading and multiply the color with it
	finalColor = finalColor * float4(LightColor, CloseToCameraFading * CloseToBackgroundFading);
	
	// Show the final color with fog on R,G,B channels
	return float4((lerp(finalColor,FogColor,saturate(length(input.Pos_CameraSpace) / FogDistance) * MaxFogIntensity)).xyz,finalColor.w);
}
