
// Used to render UV coordinates to a map for roads.
// Another shader will sample a vertical road texture using the UV coordinates drawn with this shader.

#define START_V Arg[0].x
#define END_V Arg[0].y

// You may modify this to move more data from the vertex shader to the pixel shader.
// It needs at least the position to work.
struct VS_to_PS {
	float4 Pos : SV_POSITION;
	float2 Color : COLOR;
};

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
VS_to_PS VS( VS_structure input ) {
	VS_to_PS output = (VS_to_PS)0;
	
	// Convert vertice positions and normals from bone space to object space
	float4x4 BoneToObject;
	float4 Pos_ObjectSpace;
	float3 Normal_ObjectSpace;
	int BoneIndex;
	BoneIndex = floor(input.BoneData.w);
	if (BoneIndex > -1) {
		// Get the axis system from the bone
		Engine_GetBoneMatrix(BoneIndex,BoneToObject);
		
		// Convert vertice positions from bone space to object space
		Pos_ObjectSpace = mul(float4(input.BoneData.xyz,1.0f),BoneToObject);
		
		// Transform normals from bone space to object space
		Normal_ObjectSpace = mul(input.Normal,(float3x3)BoneToObject);
	} else {
		// Take the static object space position
		Pos_ObjectSpace = input.Pos;
		
		// Copy normals
		Normal_ObjectSpace = input.Normal;
	}
	
	// Convert vertice positions from object space to 2D projection
	output.Pos = mul( mul( Pos_ObjectSpace, ObjectToWorld ), WorldToImage );
	
	// Give the vertex color multiplied with the instance color to the pixel shader
	output.Color = float2((input.Pos.x + 1.0f) / 2.0f,(1.0f - input.Pos.z) / 2.0f);
	
	return output;
}

//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS( VS_to_PS input) : SV_Target {
	// Lerp from start to end V and remove the integer part
	float V = frac(lerp(START_V,END_V,input.Color.g));
	
	// N to N+1 goes from 0 to 1 and back to 0 in a ping pong pattern
	float PPV;
	if (V < 0.5f) {
		PPV = V * 2.0f;
	} else {
		PPV = (1.0f - V) * 2.0f;
	}
	
	return float4(input.Color.r,PPV,input.Color.r,PPV);
}
