
// You may modify this to move more data from the vertex shader to the pixel shader.
// It needs at least the position to work.
struct VS_to_PS {
	float4 Pos : SV_POSITION;
	float4 Color : COLOR;
	float3 Normal_WorldSpace : NORMAL;
	float4 Pos_WorldSpace : TEXCOORD1;
	float4 Pos_CameraSpace : TEXCOORD2;
};

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
VS_to_PS VS( VS_structure input ) {
	VS_to_PS output = (VS_to_PS)0;
	
	// Convert vertice positions from object space to world space
	output.Pos_WorldSpace = mul( input.Pos, ObjectToWorld );
	// Convert vertice positions from world space to camera space
	output.Pos_CameraSpace = mul( output.Pos_WorldSpace, WorldToCamera );
	// Convert vertice positions from camera space to 2D projection
	output.Pos = mul( output.Pos_CameraSpace, CameraToImage );
	
	// Transform normals to world space
	output.Normal_WorldSpace = mul(input.Normal,(float3x3)ObjectToWorld);
	
	// Give the vertex color multiplied with the instance color to the pixel shader
	output.Color = lerp(input.Color * InstanceColor, float4(0.0f,0.0f,1.0f,1.0f), input.Selected);
	
	return output;
}

//tex_0 is the reflection map

//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS( VS_to_PS input) : SV_Target {
	// If the camera has a world space cutting plane, clip on the negative side.
	UseCuttingPlane(input.Pos_WorldSpace)
	
	// Get the normalized normal
	float3 Normal = normalize(input.Normal_WorldSpace);
	
	// Get the reflected color
	float3 Direction = reflect(normalize(CameraPos_WorldSpace - input.Pos_WorldSpace.xyz),Normal);
	float4 ImageX = tex_0.Sample( samAnisotropicMipmap, Direction.yz / (Direction.x * 4.0f), 0 );
	float4 ImageY = tex_0.Sample( samAnisotropicMipmap, Direction.xz / (Direction.y * 4.0f), 0 );
	float4 ImageZ = tex_0.Sample( samAnisotropicMipmap, Direction.xy / (Direction.z * 4.0f), 0 );
	float3 Weights = saturate((abs(Direction) * 1.5f) - 0.5f);
	Weights = Weights / (Weights.x + Weights.y + Weights.z);
	float4 ReflectionColor = (ImageX * Weights.x) + (ImageY * Weights.y) + (ImageZ * Weights.z);
	
	// Get the global light
	float3 DiffuseLight;
	float3 SpecularLight;
	Engine_GetDiffuseAndSpecularLight(input.Pos_WorldSpace,Normal,60,DiffuseLight,SpecularLight);
	float4 finalColor = ((ReflectionColor * float4(DiffuseLight,1.0f)) + float4(SpecularLight * 5.0f,1.0f)) * input.Color;
	
	// Show the final color with fog on R,G,B channels
	return float4((lerp(finalColor,FogColor,saturate(length(input.Pos_CameraSpace) / FogDistance) * MaxFogIntensity)).xyz,finalColor.w);
}
