
// You may modify this to move more data from the vertex shader to the pixel shader.
// It needs at least the position to work.
struct VS_to_PS {
	float4 Pos : SV_POSITION;
	float4 Tex : TEXCOORD0;
	float4 Color : COLOR;
	float3 Normal_WorldSpace : NORMAL;
	float4 Pos_WorldSpace : TEXCOORD1;
	float4 Pos_CameraSpace : TEXCOORD2;
};

//--------------------------------------------------------------------------------------
// Vertex Shader
//--------------------------------------------------------------------------------------
VS_to_PS VS( VS_structure input ) {
	VS_to_PS output = (VS_to_PS)0;
	
	// Convert vertice positions from object space to world space
	output.Pos_WorldSpace = mul( input.Pos, ObjectToWorld );
	// Convert vertice positions from world space to camera space
	output.Pos_CameraSpace = mul( output.Pos_WorldSpace, WorldToCamera );
	// Convert vertice positions from camera space to 2D projection
	output.Pos = mul( output.Pos_CameraSpace, CameraToImage );
	
	// Transform normals to world space
	output.Normal_WorldSpace = mul(input.Normal,(float3x3)ObjectToWorld);
	
	// Give the texture coordinates to the pixel shader
	output.Tex = input.Tex;
	
	// Give the vertex color multiplied with the instance color to the pixel shader
	output.Color = lerp(input.Color * InstanceColor, float4(0.0f,0.0f,1.0f,1.0f), input.Selected);
	
	return output;
}

#define WaveOffset1 Arg[0].xy
#define WaveOffset2 Arg[0].zw
#define WaveScale Arg[1].x
#define WaveIntensity Arg[1].y

#define ReflectionTexture tex_0
#define WaveTexture tex_2
#define DiffuseTexture tex_3

//--------------------------------------------------------------------------------------
// Pixel Shader
//--------------------------------------------------------------------------------------
float4 PS( VS_to_PS input) : SV_Target {
	float Foglevel = saturate(length(input.Pos_CameraSpace) / FogDistance) * MaxFogIntensity;
	
	// Sample the waves multiple times with different offsets and add the results before normalizing
	float2 distortOffset = (WaveTexture.Sample( samSharp, (input.Pos_WorldSpace.xz + WaveOffset1) / WaveScale ).rg
			   + WaveTexture.Sample( samSharp, (input.Pos_WorldSpace.xz + WaveOffset2) / WaveScale ).rg
			   - 1.0f);
	
	float2 waveOffset = distortOffset * (WaveIntensity / length(input.Pos_CameraSpace));
	
	// Sample the reflection image
	float3 reflectionColor = ReflectionTexture.Sample( samClampedSharp, float2(input.Pos.x / Dimensions.x, 1.0f - (input.Pos.y / Dimensions.y)) + (waveOffset / float2(CameraXScale,CameraYScale) * 0.15f) ).rgb;
	
	// Create the normal for light
	float3 Normal = normalize(float3(waveOffset.x, 1.0f, waveOffset.y));
	
	// Only apply diffuse light to diffuse textures and specular light to reflection texture
	float3 DiffuseLight;
	float3 SpecularLight;
	Engine_GetDiffuseAndSpecularLight(input.Pos_WorldSpace.xyz,normalize(Normal),20,DiffuseLight,SpecularLight);
	
	// Sample diffuse texture
	float3 DiffuseColor = DiffuseTexture.Sample( samAnisotropicMipmap, (input.Pos_WorldSpace.xz * 0.1f) + (distortOffset * 0.02f) ).rgb;
	
	// Use diffuse light on the diffuse texture
	DiffuseColor = DiffuseColor * DiffuseLight;
	
	// Apply fog to the diffuse texture
	DiffuseColor = lerp(DiffuseColor,FogColor,Foglevel);
	
	// If a sun is rendered in the reflection, this won't be needed.
	reflectionColor = reflectionColor + (float4(SpecularLight,1.0f) * (1.0f - Foglevel));
	
	// Calculate how much you see thru the water
	float RefractionRatio = saturate(normalize(CameraPos_WorldSpace - input.Pos_WorldSpace).y);
	
	// Mix reflection and diffuse
	return float4(lerp(reflectionColor,DiffuseColor,RefractionRatio),1.0f);
}
